import { z } from "zod"
import DOMPurify from "dompurify"

const sanitize = (value: string) => {
  if (typeof window !== "undefined") {
    return DOMPurify.sanitize(value.trim())
  }
  return value.trim()
}

export const sendMoneySchema = z.object({
  amount: z.number().min(1, "Amount must be at least 1").max(10000000, "Amount cannot exceed 1 crore"),
  referenceNumber: z
    .string()
    .min(5, "Reference number must be at least 5 characters")
    .max(50, "Reference number is too long")
    .transform(sanitize),
})

export const cancelWithdrawalSchema = z.object({
  reason: z
    .string()
    .min(10, "Reason must be at least 10 characters")
    .max(500, "Reason is too long")
    .transform(sanitize),
})

export type SendMoneyInput = z.infer<typeof sendMoneySchema>
export type CancelWithdrawalInput = z.infer<typeof cancelWithdrawalSchema>
