import { vehicleDataStore } from "./vehicles"
import { locationDataStore } from "./locations"

export type VehicleBookingStatus = "booked" | "ongoing" | "completed" | "cancelled"

export interface VehicleBooking {
  id: string
  customerId: string
  customerName: string
  bookingDate: string
  journeyDate: string
  originStateId: string
  originStateName: string
  originCityId: string
  originCityName: string
  destinationStateId: string
  destinationStateName: string
  destinationCityId: string
  destinationCityName: string
  vehicleId: string
  vehicleType: string
  bookingAmount: number
  status: VehicleBookingStatus
  createdAt: string
  updatedAt: string
}

// In-memory vehicle booking data store
class VehicleBookingDataStore {
  private bookings: Map<string, VehicleBooking> = new Map()
  private idCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const vehicles = vehicleDataStore.getAllVehicles()
    const cities = locationDataStore.getAllCities()
    const activeCities = cities.filter((c) => c.status === "active")

    const customerNames = [
      "Ramesh Kumar",
      "Sunita Devi",
      "Anil Sharma",
      "Kavita Patel",
      "Suresh Yadav",
      "Meena Singh",
      "Rajiv Gupta",
      "Anita Verma",
      "Vinod Mishra",
      "Geeta Rani",
      "Prakash Joshi",
      "Rekha Agarwal",
      "Mohan Lal",
      "Saroj Kumari",
      "Dinesh Choudhary",
      "Usha Devi",
      "Kishan Singh",
      "Poonam Sharma",
      "Ravi Prasad",
      "Seema Gupta",
      "Ashok Kumar",
      "Neelam Devi",
      "Sanjay Verma",
      "Shanti Devi",
      "Deepak Rao",
    ]

    const sampleBookings: Array<{
      customerName: string
      vehicleIndex: number
      originCityIndex: number
      destCityIndex: number
      bookingDate: string
      journeyDate: string
      bookingAmount: number
      status: VehicleBookingStatus
    }> = [
      // Booked (upcoming)
      {
        customerName: customerNames[0],
        vehicleIndex: 0,
        originCityIndex: 0,
        destCityIndex: 1,
        bookingDate: "2024-02-01",
        journeyDate: "2024-02-20",
        bookingAmount: 4500,
        status: "booked",
      },
      {
        customerName: customerNames[1],
        vehicleIndex: 1,
        originCityIndex: 2,
        destCityIndex: 3,
        bookingDate: "2024-02-02",
        journeyDate: "2024-02-22",
        bookingAmount: 8500,
        status: "booked",
      },
      {
        customerName: customerNames[2],
        vehicleIndex: 2,
        originCityIndex: 0,
        destCityIndex: 4,
        bookingDate: "2024-02-03",
        journeyDate: "2024-02-25",
        bookingAmount: 5200,
        status: "booked",
      },
      {
        customerName: customerNames[3],
        vehicleIndex: 3,
        originCityIndex: 1,
        destCityIndex: 2,
        bookingDate: "2024-02-04",
        journeyDate: "2024-02-28",
        bookingAmount: 12000,
        status: "booked",
      },
      {
        customerName: customerNames[4],
        vehicleIndex: 4,
        originCityIndex: 3,
        destCityIndex: 0,
        bookingDate: "2024-02-05",
        journeyDate: "2024-03-01",
        bookingAmount: 6800,
        status: "booked",
      },
      {
        customerName: customerNames[5],
        vehicleIndex: 5,
        originCityIndex: 4,
        destCityIndex: 1,
        bookingDate: "2024-02-06",
        journeyDate: "2024-03-05",
        bookingAmount: 3200,
        status: "booked",
      },

      // Ongoing (in progress)
      {
        customerName: customerNames[6],
        vehicleIndex: 0,
        originCityIndex: 1,
        destCityIndex: 3,
        bookingDate: "2024-01-25",
        journeyDate: "2024-02-10",
        bookingAmount: 5800,
        status: "ongoing",
      },
      {
        customerName: customerNames[7],
        vehicleIndex: 2,
        originCityIndex: 0,
        destCityIndex: 2,
        bookingDate: "2024-01-26",
        journeyDate: "2024-02-11",
        bookingAmount: 7200,
        status: "ongoing",
      },
      {
        customerName: customerNames[8],
        vehicleIndex: 3,
        originCityIndex: 2,
        destCityIndex: 4,
        bookingDate: "2024-01-27",
        journeyDate: "2024-02-12",
        bookingAmount: 15500,
        status: "ongoing",
      },
      {
        customerName: customerNames[9],
        vehicleIndex: 1,
        originCityIndex: 4,
        destCityIndex: 0,
        bookingDate: "2024-01-28",
        journeyDate: "2024-02-13",
        bookingAmount: 9800,
        status: "ongoing",
      },
      {
        customerName: customerNames[10],
        vehicleIndex: 4,
        originCityIndex: 3,
        destCityIndex: 1,
        bookingDate: "2024-01-29",
        journeyDate: "2024-02-14",
        bookingAmount: 4200,
        status: "ongoing",
      },

      // Completed (past)
      {
        customerName: customerNames[11],
        vehicleIndex: 0,
        originCityIndex: 0,
        destCityIndex: 3,
        bookingDate: "2024-01-01",
        journeyDate: "2024-01-10",
        bookingAmount: 6500,
        status: "completed",
      },
      {
        customerName: customerNames[12],
        vehicleIndex: 1,
        originCityIndex: 1,
        destCityIndex: 4,
        bookingDate: "2024-01-02",
        journeyDate: "2024-01-12",
        bookingAmount: 11200,
        status: "completed",
      },
      {
        customerName: customerNames[13],
        vehicleIndex: 2,
        originCityIndex: 2,
        destCityIndex: 0,
        bookingDate: "2024-01-03",
        journeyDate: "2024-01-14",
        bookingAmount: 4800,
        status: "completed",
      },
      {
        customerName: customerNames[14],
        vehicleIndex: 3,
        originCityIndex: 3,
        destCityIndex: 1,
        bookingDate: "2024-01-04",
        journeyDate: "2024-01-15",
        bookingAmount: 13500,
        status: "completed",
      },
      {
        customerName: customerNames[15],
        vehicleIndex: 4,
        originCityIndex: 4,
        destCityIndex: 2,
        bookingDate: "2024-01-05",
        journeyDate: "2024-01-18",
        bookingAmount: 5600,
        status: "completed",
      },
      {
        customerName: customerNames[16],
        vehicleIndex: 5,
        originCityIndex: 0,
        destCityIndex: 4,
        bookingDate: "2024-01-06",
        journeyDate: "2024-01-20",
        bookingAmount: 3800,
        status: "completed",
      },
      {
        customerName: customerNames[17],
        vehicleIndex: 0,
        originCityIndex: 1,
        destCityIndex: 0,
        bookingDate: "2024-01-07",
        journeyDate: "2024-01-22",
        bookingAmount: 7800,
        status: "completed",
      },
      {
        customerName: customerNames[18],
        vehicleIndex: 1,
        originCityIndex: 2,
        destCityIndex: 3,
        bookingDate: "2024-01-08",
        journeyDate: "2024-01-25",
        bookingAmount: 9200,
        status: "completed",
      },

      // Cancelled
      {
        customerName: customerNames[19],
        vehicleIndex: 0,
        originCityIndex: 0,
        destCityIndex: 1,
        bookingDate: "2024-01-10",
        journeyDate: "2024-01-28",
        bookingAmount: 0,
        status: "cancelled",
      },
      {
        customerName: customerNames[20],
        vehicleIndex: 2,
        originCityIndex: 2,
        destCityIndex: 4,
        bookingDate: "2024-01-12",
        journeyDate: "2024-01-30",
        bookingAmount: 0,
        status: "cancelled",
      },
      {
        customerName: customerNames[21],
        vehicleIndex: 3,
        originCityIndex: 3,
        destCityIndex: 0,
        bookingDate: "2024-01-14",
        journeyDate: "2024-02-02",
        bookingAmount: 0,
        status: "cancelled",
      },
      {
        customerName: customerNames[22],
        vehicleIndex: 1,
        originCityIndex: 1,
        destCityIndex: 3,
        bookingDate: "2024-01-16",
        journeyDate: "2024-02-05",
        bookingAmount: 0,
        status: "cancelled",
      },
      {
        customerName: customerNames[23],
        vehicleIndex: 4,
        originCityIndex: 4,
        destCityIndex: 2,
        bookingDate: "2024-01-18",
        journeyDate: "2024-02-08",
        bookingAmount: 0,
        status: "cancelled",
      },
    ]

    sampleBookings.forEach((booking) => {
      const vehicle = vehicles[booking.vehicleIndex % vehicles.length]
      const originCity = activeCities[booking.originCityIndex % activeCities.length]
      const destCity = activeCities[booking.destCityIndex % activeCities.length]
      const originState = locationDataStore.getStateById(originCity?.stateId || "")
      const destState = locationDataStore.getStateById(destCity?.stateId || "")

      if (vehicle && originCity && destCity && originState && destState) {
        this.idCounter++
        const id = `vb_${String(this.idCounter).padStart(4, "0")}`
        const now = new Date().toISOString().split("T")[0]

        this.bookings.set(id, {
          id,
          customerId: `cust_${String(this.idCounter).padStart(3, "0")}`,
          customerName: booking.customerName,
          bookingDate: booking.bookingDate,
          journeyDate: booking.journeyDate,
          originStateId: originState.id,
          originStateName: originState.name,
          originCityId: originCity.id,
          originCityName: originCity.name,
          destinationStateId: destState.id,
          destinationStateName: destState.name,
          destinationCityId: destCity.id,
          destinationCityName: destCity.name,
          vehicleId: vehicle.id,
          vehicleType: `${vehicle.categoryName} - ${vehicle.name}`,
          bookingAmount: booking.bookingAmount,
          status: booking.status,
          createdAt: now,
          updatedAt: now,
        })
      }
    })
  }

  getAllBookings(): VehicleBooking[] {
    return Array.from(this.bookings.values()).sort(
      (a, b) => new Date(b.journeyDate).getTime() - new Date(a.journeyDate).getTime(),
    )
  }

  getBookingsByStatus(status: VehicleBookingStatus): VehicleBooking[] {
    return this.getAllBookings().filter((b) => b.status === status)
  }

  getBookedBookings(): VehicleBooking[] {
    return this.getBookingsByStatus("booked")
  }

  getOngoingBookings(): VehicleBooking[] {
    return this.getBookingsByStatus("ongoing")
  }

  getCompletedBookings(): VehicleBooking[] {
    return this.getBookingsByStatus("completed")
  }

  getCancelledBookings(): VehicleBooking[] {
    return this.getBookingsByStatus("cancelled")
  }

  getBookingById(id: string): VehicleBooking | undefined {
    return this.bookings.get(id)
  }

  getBookingStats(): {
    booked: number
    ongoing: number
    completed: number
    cancelled: number
    total: number
    bookedAmount: number
    ongoingAmount: number
    completedAmount: number
    totalAmount: number
  } {
    const all = this.getAllBookings()
    const booked = all.filter((b) => b.status === "booked")
    const ongoing = all.filter((b) => b.status === "ongoing")
    const completed = all.filter((b) => b.status === "completed")
    const cancelled = all.filter((b) => b.status === "cancelled")

    return {
      booked: booked.length,
      ongoing: ongoing.length,
      completed: completed.length,
      cancelled: cancelled.length,
      total: all.length,
      bookedAmount: booked.reduce((sum, b) => sum + b.bookingAmount, 0),
      ongoingAmount: ongoing.reduce((sum, b) => sum + b.bookingAmount, 0),
      completedAmount: completed.reduce((sum, b) => sum + b.bookingAmount, 0),
      totalAmount: all.reduce((sum, b) => sum + b.bookingAmount, 0),
    }
  }
}

export const vehicleBookingDataStore = new VehicleBookingDataStore()
