export type EarningUserType = "agent" | "hotel"

export interface EarningRecord {
  id: string
  userType: EarningUserType
  name: string
  email: string
  phone: string
  totalEarning: number
  thisMonthEarning: number
  walletBalance: number
  totalWithdrawn: number
  joinedAt: string
  lastActiveAt: string
}

// In-memory earnings data store
class EarningsDataStore {
  private earnings: Map<string, EarningRecord> = new Map()

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const sampleEarnings: EarningRecord[] = [
      // Agent earnings
      {
        id: "earn_agent_001",
        userType: "agent",
        name: "Rajesh Travel Services",
        email: "rajesh.travels@email.com",
        phone: "+91 9876543210",
        totalEarning: 485000,
        thisMonthEarning: 42500,
        walletBalance: 125000,
        totalWithdrawn: 360000,
        joinedAt: "2023-03-15",
        lastActiveAt: "2024-12-06",
      },
      {
        id: "earn_agent_002",
        userType: "agent",
        name: "Priya Tours & Travels",
        email: "priya.tours@email.com",
        phone: "+91 9876543211",
        totalEarning: 328000,
        thisMonthEarning: 28000,
        walletBalance: 78000,
        totalWithdrawn: 250000,
        joinedAt: "2023-06-20",
        lastActiveAt: "2024-12-05",
      },
      {
        id: "earn_agent_003",
        userType: "agent",
        name: "Amit Pilgrimage Services",
        email: "amit.pilgrimage@email.com",
        phone: "+91 9876543212",
        totalEarning: 715000,
        thisMonthEarning: 65000,
        walletBalance: 215000,
        totalWithdrawn: 500000,
        joinedAt: "2022-11-10",
        lastActiveAt: "2024-12-06",
      },
      {
        id: "earn_agent_004",
        userType: "agent",
        name: "Sunita Yatra Agency",
        email: "sunita.yatra@email.com",
        phone: "+91 9876543213",
        totalEarning: 152000,
        thisMonthEarning: 18000,
        walletBalance: 32000,
        totalWithdrawn: 120000,
        joinedAt: "2024-02-05",
        lastActiveAt: "2024-12-04",
      },
      {
        id: "earn_agent_005",
        userType: "agent",
        name: "Vikram Holy Tours",
        email: "vikram.holy@email.com",
        phone: "+91 9876543214",
        totalEarning: 392000,
        thisMonthEarning: 35000,
        walletBalance: 92000,
        totalWithdrawn: 300000,
        joinedAt: "2023-08-12",
        lastActiveAt: "2024-12-06",
      },
      {
        id: "earn_agent_006",
        userType: "agent",
        name: "Ganesh Tirth Yatra",
        email: "ganesh.yatra@email.com",
        phone: "+91 9876543215",
        totalEarning: 568000,
        thisMonthEarning: 48000,
        walletBalance: 168000,
        totalWithdrawn: 400000,
        joinedAt: "2023-01-25",
        lastActiveAt: "2024-12-05",
      },
      {
        id: "earn_agent_007",
        userType: "agent",
        name: "Shanti Travel Solutions",
        email: "shanti.travels@email.com",
        phone: "+91 9876543216",
        totalEarning: 245000,
        thisMonthEarning: 22000,
        walletBalance: 45000,
        totalWithdrawn: 200000,
        joinedAt: "2023-10-08",
        lastActiveAt: "2024-12-03",
      },
      {
        id: "earn_agent_008",
        userType: "agent",
        name: "Divine Path Tours",
        email: "divine.path@email.com",
        phone: "+91 9876543217",
        totalEarning: 623000,
        thisMonthEarning: 55000,
        walletBalance: 123000,
        totalWithdrawn: 500000,
        joinedAt: "2022-08-30",
        lastActiveAt: "2024-12-06",
      },
      {
        id: "earn_agent_009",
        userType: "agent",
        name: "Om Shri Travels",
        email: "om.shri@email.com",
        phone: "+91 9876543218",
        totalEarning: 189000,
        thisMonthEarning: 15000,
        walletBalance: 39000,
        totalWithdrawn: 150000,
        joinedAt: "2024-04-18",
        lastActiveAt: "2024-12-02",
      },
      {
        id: "earn_agent_010",
        userType: "agent",
        name: "Bhakti Yatra Services",
        email: "bhakti.yatra@email.com",
        phone: "+91 9876543219",
        totalEarning: 412000,
        thisMonthEarning: 38000,
        walletBalance: 112000,
        totalWithdrawn: 300000,
        joinedAt: "2023-05-14",
        lastActiveAt: "2024-12-05",
      },
      // Hotel earnings
      {
        id: "earn_hotel_001",
        userType: "hotel",
        name: "Divine Stay Resort",
        email: "divine.stay@email.com",
        phone: "+91 9988776655",
        totalEarning: 1850000,
        thisMonthEarning: 165000,
        walletBalance: 450000,
        totalWithdrawn: 1400000,
        joinedAt: "2022-06-10",
        lastActiveAt: "2024-12-06",
      },
      {
        id: "earn_hotel_002",
        userType: "hotel",
        name: "Pilgrim Palace Hotel",
        email: "pilgrim.palace@email.com",
        phone: "+91 9988776656",
        totalEarning: 1320000,
        thisMonthEarning: 120000,
        walletBalance: 320000,
        totalWithdrawn: 1000000,
        joinedAt: "2022-09-22",
        lastActiveAt: "2024-12-05",
      },
      {
        id: "earn_hotel_003",
        userType: "hotel",
        name: "Spiritual Inn",
        email: "spiritual.inn@email.com",
        phone: "+91 9988776657",
        totalEarning: 780000,
        thisMonthEarning: 72000,
        walletBalance: 180000,
        totalWithdrawn: 600000,
        joinedAt: "2023-02-14",
        lastActiveAt: "2024-12-06",
      },
      {
        id: "earn_hotel_004",
        userType: "hotel",
        name: "Sacred Stay Lodge",
        email: "sacred.stay@email.com",
        phone: "+91 9988776658",
        totalEarning: 495000,
        thisMonthEarning: 45000,
        walletBalance: 95000,
        totalWithdrawn: 400000,
        joinedAt: "2023-07-08",
        lastActiveAt: "2024-12-04",
      },
      {
        id: "earn_hotel_005",
        userType: "hotel",
        name: "Holy Heritage Hotel",
        email: "holy.heritage@email.com",
        phone: "+91 9988776659",
        totalEarning: 1175000,
        thisMonthEarning: 105000,
        walletBalance: 275000,
        totalWithdrawn: 900000,
        joinedAt: "2022-12-05",
        lastActiveAt: "2024-12-05",
      },
      {
        id: "earn_hotel_006",
        userType: "hotel",
        name: "Moksha Retreat",
        email: "moksha.retreat@email.com",
        phone: "+91 9988776660",
        totalEarning: 2140000,
        thisMonthEarning: 195000,
        walletBalance: 540000,
        totalWithdrawn: 1600000,
        joinedAt: "2022-04-18",
        lastActiveAt: "2024-12-06",
      },
      {
        id: "earn_hotel_007",
        userType: "hotel",
        name: "Tirtha Residency",
        email: "tirtha.residency@email.com",
        phone: "+91 9988776661",
        totalEarning: 925000,
        thisMonthEarning: 85000,
        walletBalance: 225000,
        totalWithdrawn: 700000,
        joinedAt: "2023-03-22",
        lastActiveAt: "2024-12-05",
      },
      {
        id: "earn_hotel_008",
        userType: "hotel",
        name: "Yatri Bhavan",
        email: "yatri.bhavan@email.com",
        phone: "+91 9988776662",
        totalEarning: 685000,
        thisMonthEarning: 62000,
        walletBalance: 135000,
        totalWithdrawn: 550000,
        joinedAt: "2023-06-30",
        lastActiveAt: "2024-12-04",
      },
      {
        id: "earn_hotel_009",
        userType: "hotel",
        name: "Punya Stay Hotel",
        email: "punya.stay@email.com",
        phone: "+91 9988776663",
        totalEarning: 548000,
        thisMonthEarning: 48000,
        walletBalance: 98000,
        totalWithdrawn: 450000,
        joinedAt: "2023-09-15",
        lastActiveAt: "2024-12-03",
      },
      {
        id: "earn_hotel_010",
        userType: "hotel",
        name: "Dharma Guest House",
        email: "dharma.guest@email.com",
        phone: "+91 9988776664",
        totalEarning: 412000,
        thisMonthEarning: 38000,
        walletBalance: 62000,
        totalWithdrawn: 350000,
        joinedAt: "2024-01-20",
        lastActiveAt: "2024-12-02",
      },
    ]

    sampleEarnings.forEach((earning) => {
      this.earnings.set(earning.id, earning)
    })
  }

  getEarningsByType(userType: EarningUserType): EarningRecord[] {
    return Array.from(this.earnings.values())
      .filter((e) => e.userType === userType)
      .sort((a, b) => b.totalEarning - a.totalEarning)
  }

  getAgentEarnings(): EarningRecord[] {
    return this.getEarningsByType("agent")
  }

  getHotelEarnings(): EarningRecord[] {
    return this.getEarningsByType("hotel")
  }

  getEarningById(id: string): EarningRecord | undefined {
    return this.earnings.get(id)
  }

  getTotalEarningsByType(userType: EarningUserType): {
    totalEarning: number
    thisMonthEarning: number
    totalWalletBalance: number
    totalWithdrawn: number
  } {
    const records = this.getEarningsByType(userType)
    return records.reduce(
      (acc, record) => ({
        totalEarning: acc.totalEarning + record.totalEarning,
        thisMonthEarning: acc.thisMonthEarning + record.thisMonthEarning,
        totalWalletBalance: acc.totalWalletBalance + record.walletBalance,
        totalWithdrawn: acc.totalWithdrawn + record.totalWithdrawn,
      }),
      { totalEarning: 0, thisMonthEarning: 0, totalWalletBalance: 0, totalWithdrawn: 0 },
    )
  }
}

export const earningsDataStore = new EarningsDataStore()
