"use client"

interface SessionData {
  userId?: string
  email: string
  role: "admin" | "customer" | "agent" | "hotel"
  name?: string
  loginTime?: string
}

/**
 * Validate client-side session data structure
 */
export function validateSessionData(data: unknown): data is SessionData {
  if (!data || typeof data !== "object") {
    return false
  }

  const session = data as Record<string, unknown>

  return (
    typeof session.email === "string" &&
    session.email.length > 0 &&
    typeof session.role === "string" &&
    ["admin", "customer", "agent", "hotel"].includes(session.role)
  )
}

/**
 * Get client session from sessionStorage with validation
 */
export function getClientSession(storageKey: string): SessionData | null {
  if (typeof window === "undefined") {
    return null
  }

  try {
    const stored = sessionStorage.getItem(storageKey)
    if (!stored) {
      return null
    }

    const parsed = JSON.parse(stored)
    if (!validateSessionData(parsed)) {
      sessionStorage.removeItem(storageKey)
      return null
    }

    return parsed
  } catch {
    sessionStorage.removeItem(storageKey)
    return null
  }
}

/**
 * Set client session in sessionStorage
 */
export function setClientSession(storageKey: string, data: SessionData): void {
  if (typeof window === "undefined") {
    return
  }

  try {
    sessionStorage.setItem(storageKey, JSON.stringify(data))
  } catch {
    console.error("Failed to set client session")
  }
}

/**
 * Clear client session from sessionStorage
 */
export function clearClientSession(storageKey: string): void {
  if (typeof window === "undefined") {
    return
  }

  try {
    sessionStorage.removeItem(storageKey)
  } catch {
    console.error("Failed to clear client session")
  }
}

export type { SessionData }
