"use client"

import { useState, useMemo } from "react"
import { useRouter } from "next/navigation"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Plus } from "lucide-react"
import { HotelPartnerTable } from "./hotel-partner-table"
import { ViewDocumentsDialog } from "./view-documents-dialog"
import { DeletePartnerDialog } from "./delete-partner-dialog"
import { hotelPartnerDataStore, type HotelPartner } from "@/lib/data/hotel-partners"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"

export function HotelPartnersClient() {
  const router = useRouter()
  const toast = useToastEnhanced()
  const [partners, setPartners] = useState<HotelPartner[]>(hotelPartnerDataStore.getAllPartners())
  const [activeTab, setActiveTab] = useState("all")

  // Dialog states
  const [selectedPartner, setSelectedPartner] = useState<HotelPartner | null>(null)
  const [isDocumentsOpen, setIsDocumentsOpen] = useState(false)
  const [isDeleteOpen, setIsDeleteOpen] = useState(false)
  const [partnerToDelete, setPartnerToDelete] = useState<HotelPartner | null>(null)

  const activePartners = useMemo(() => partners.filter((p) => p.status === "active"), [partners])
  const inactivePartners = useMemo(() => partners.filter((p) => p.status === "inactive"), [partners])

  const displayedPartners = useMemo(() => {
    switch (activeTab) {
      case "active":
        return activePartners
      case "inactive":
        return inactivePartners
      default:
        return partners
    }
  }, [activeTab, partners, activePartners, inactivePartners])

  const handleViewDocuments = (partner: HotelPartner) => {
    setSelectedPartner(partner)
    setIsDocumentsOpen(true)
  }

  const handleEdit = (partner: HotelPartner) => {
    router.push(`/admin/hotel-partner/add-new?id=${partner.id}`)
  }

  const handleDeleteClick = (partner: HotelPartner) => {
    setPartnerToDelete(partner)
    setIsDeleteOpen(true)
  }

  const handleDeleteConfirm = () => {
    if (!partnerToDelete) return

    const deleted = hotelPartnerDataStore.deletePartner(partnerToDelete.id)
    if (deleted) {
      setPartners(hotelPartnerDataStore.getAllPartners())
      toast.success({
        title: "Partner Deleted",
        description: `${partnerToDelete.hotelName} has been deleted successfully.`,
      })
    }
    setIsDeleteOpen(false)
    setPartnerToDelete(null)
  }

  const handleToggleStatus = (partner: HotelPartner) => {
    const updated = hotelPartnerDataStore.toggleStatus(partner.id)
    if (updated) {
      setPartners(hotelPartnerDataStore.getAllPartners())
      toast.success({
        title: `Partner ${updated.status === "active" ? "Activated" : "Deactivated"}`,
        description: `${partner.hotelName} is now ${updated.status}.`,
      })
    }
  }

  const handleAddNew = () => {
    router.push("/admin/hotel-partner/add-new")
  }

  return (
    <section aria-labelledby="hotel-partners-heading" className="admin-section">
      <header className="admin-page-header flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <div>
          <h1 id="hotel-partners-heading" className="admin-page-title">
            Hotel Partners
          </h1>
          <p className="admin-page-description">Manage approved hotel partners on the platform.</p>
        </div>
        <Button onClick={handleAddNew} className="w-full sm:w-auto">
          <Plus className="mr-2 h-4 w-4" aria-hidden="true" />
          Add New Partner
        </Button>
      </header>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50">
          <TabsTrigger
            value="all"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            All ({partners.length})
          </TabsTrigger>
          <TabsTrigger
            value="active"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Active ({activePartners.length})
          </TabsTrigger>
          <TabsTrigger
            value="inactive"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Inactive ({inactivePartners.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="all" className="mt-0">
          <HotelPartnerTable
            partners={displayedPartners}
            onViewDocuments={handleViewDocuments}
            onEdit={handleEdit}
            onDelete={handleDeleteClick}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="active" className="mt-0">
          <HotelPartnerTable
            partners={displayedPartners}
            onViewDocuments={handleViewDocuments}
            onEdit={handleEdit}
            onDelete={handleDeleteClick}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="inactive" className="mt-0">
          <HotelPartnerTable
            partners={displayedPartners}
            onViewDocuments={handleViewDocuments}
            onEdit={handleEdit}
            onDelete={handleDeleteClick}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>
      </Tabs>

      {/* Dialogs */}
      <ViewDocumentsDialog partner={selectedPartner} open={isDocumentsOpen} onOpenChange={setIsDocumentsOpen} />

      <DeletePartnerDialog
        partner={partnerToDelete}
        open={isDeleteOpen}
        onOpenChange={setIsDeleteOpen}
        onConfirm={handleDeleteConfirm}
      />
    </section>
  )
}
