import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { generatePageMetadata, generateJsonLd } from "@/lib/seo/metadata"
import type { Metadata } from "next"
import Link from "next/link"
import Image from "next/image"
import {
  Star,
  Users,
  Shield,
  Clock,
  MapPin,
  ArrowRight,
  Heart,
  TrendingUp,
  Headphones,
  CreditCard,
  Award,
  Fuel,
  Wifi,
  Coffee,
  ParkingCircle,
} from "lucide-react"
import { HeroSlider } from "@/components/frontend/hero-slider"

export const metadata: Metadata = generatePageMetadata({
  title: "Home",
  description:
    "Experience seamless travel with Teertham - Book vehicles, hotels, spiritual packages, and custom travel experiences across India. Your journey begins here.",
  path: "/",
  type: "website",
})

export default function HomePage() {
  const jsonLd = generateJsonLd({
    "@type": "WebSite",
    name: "Teertham",
    url: process.env.NEXT_PUBLIC_SITE_URL || "https://teertham.com",
    description: "Your trusted partner for travel bookings and spiritual journeys",
  })

  const stats = [
    { icon: Users, value: "50,000+", label: "Happy Customers" },
    { icon: Star, value: "4.9/5", label: "Average Rating" },
    { icon: MapPin, value: "100+", label: "Destinations" },
    { icon: TrendingUp, value: "98%", label: "Success Rate" },
  ]

  const advantages = [
    {
      icon: Shield,
      title: "100% Secure Booking",
      description: "Your payments and personal information are always protected with bank-grade security.",
    },
    {
      icon: CreditCard,
      title: "Best Price Guarantee",
      description: "We offer the most competitive prices with no hidden charges. Best value for your money.",
    },
    {
      icon: Headphones,
      title: "24/7 Customer Support",
      description: "Our dedicated team is available round the clock to assist you with any queries.",
    },
    {
      icon: Clock,
      title: "Instant Confirmation",
      description: "Get immediate booking confirmation with all details sent directly to your email.",
    },
    {
      icon: Award,
      title: "Verified Partners",
      description: "All our hotels, vehicles, and service providers are thoroughly verified for quality.",
    },
    {
      icon: Heart,
      title: "Personalized Experience",
      description: "Customized travel packages designed to match your preferences and budget.",
    },
  ]

  const destinationCategories = [
    {
      name: "Spiritual",
      image: "/om-symbol-golden-spiritual-icon.jpg",
      destinations: [
        {
          name: "Badrinath Temple",
          image: "/badrinath-temple-uttarakhand-snow-mountains-hindu-.jpg",
          location: "Chamoli",
        },
        {
          name: "Kedarnath Temple",
          image: "/kedarnath-temple-stone-shrine-himalayan-mountains-.jpg",
          location: "Rudraprayag",
        },
        {
          name: "Gangotri Temple",
          image: "/gangotri-temple-white-marble-himalayan-backdrop-ri.jpg",
          location: "Uttarkashi",
        },
        {
          name: "Yamunotri Temple",
          image: "/yamunotri-temple-mountain-shrine-hot-springs-pilgr.jpg",
          location: "Uttarkashi",
        },
        {
          name: "Rishikesh Ganga Aarti",
          image: "/rishikesh-ganga-aarti-evening-ceremony-fire-lamps-.jpg",
          location: "Rishikesh",
        },
      ],
    },
    {
      name: "Adventure",
      image: "/mountain-adventure-hiking-icon-compass.jpg",
      destinations: [
        {
          name: "Valley of Flowers",
          image: "/valley-of-flowers-national-park-colorful-wildflowe.jpg",
          location: "Chamoli",
        },
        {
          name: "Auli Skiing",
          image: "/auli-skiing-snow-slope-skiers-winter-sport-himalay.jpg",
          location: "Auli",
        },
        {
          name: "Rishikesh Rafting",
          image: "/rishikesh-white-water-rafting-ganga-river-adventur.jpg",
          location: "Rishikesh",
        },
        {
          name: "Chopta Meadows",
          image: "/chopta-meadows-green-alpine-grassland-snow-peaks-t.jpg",
          location: "Rudraprayag",
        },
        {
          name: "Mussoorie Cable Ride",
          image: "/mussoorie-cable-car-ropeway-mountain-valley-view-s.jpg",
          location: "Mussoorie",
        },
      ],
    },
    {
      name: "Trekking",
      image: "/trekking-boots-mountains-icon.jpg",
      destinations: [
        {
          name: "Roopkund Trek",
          image: "/roopkund-trek-skeleton-lake-himalayan-snow-peaks.jpg",
          location: "Chamoli",
        },
        {
          name: "Har Ki Dun",
          image: "/har-ki-dun-valley-trek-green-meadows-snow-mountain.jpg",
          location: "Uttarkashi",
        },
        {
          name: "Kedarkantha",
          image: "/kedarkantha-trek-summit-snow-covered-peak-sunrise.jpg",
          location: "Dehradun",
        },
        {
          name: "Nag Tibba",
          image: "/nag-tibba-trek-campsite-tents-mountain-view-night.jpg",
          location: "Mussoorie",
        },
        {
          name: "Chandrashila Peak",
          image: "/chandrashila-peak-summit-360-himalayan-view-tungnath.jpg",
          location: "Chopta",
        },
      ],
    },
  ]

  const vehicles = [
    {
      name: "Swift Dzire",
      type: "Sedan",
      image: "/swift-dzire-white-sedan-car-road-trip.jpg",
      seats: 4,
      fuel: "Petrol/CNG",
      price: 2500,
      rating: 4.8,
    },
    {
      name: "Toyota Innova",
      type: "SUV",
      image: "/toyota-innova-crysta-white-suv-family-car.jpg",
      seats: 7,
      fuel: "Diesel",
      price: 4500,
      rating: 4.9,
    },
    {
      name: "Maruti Ertiga",
      type: "MPV",
      image: "/maruti-ertiga-silver-mpv-7-seater-car.jpg",
      seats: 7,
      fuel: "Petrol/CNG",
      price: 3500,
      rating: 4.7,
    },
    {
      name: "Force Traveller",
      type: "Tempo",
      image: "/force-traveller-tempo-white-12-seater-van.jpg",
      seats: 12,
      fuel: "Diesel",
      price: 6500,
      rating: 4.6,
    },
    {
      name: "Luxury Coach",
      type: "Bus",
      image: "/luxury-coach-bus-volvo-white-tourist-bus.jpg",
      seats: 25,
      fuel: "Diesel",
      price: 12000,
      rating: 4.8,
    },
  ]

  const hotels = [
    {
      name: "The Himalayan Retreat",
      location: "Rishikesh",
      image: "/himalayan-retreat-hotel-rishikesh-luxury-resort.jpg",
      price: 4500,
      rating: 4.9,
      reviews: 1250,
      amenities: ["wifi", "parking", "breakfast"],
    },
    {
      name: "Mountain View Resort",
      location: "Mussoorie",
      image: "/mountain-view-resort-mussoorie-hill-station-hotel.jpg",
      price: 6500,
      rating: 4.8,
      reviews: 980,
      amenities: ["wifi", "parking", "breakfast"],
    },
    {
      name: "Divine Stay",
      location: "Haridwar",
      image: "/divine-stay-hotel-haridwar-ganga-view-room.jpg",
      price: 2500,
      rating: 4.6,
      reviews: 750,
      amenities: ["wifi", "breakfast"],
    },
    {
      name: "Valley Residency",
      location: "Dehradun",
      image: "/valley-residency-dehradun-modern-hotel-room.jpg",
      price: 3500,
      rating: 4.7,
      reviews: 620,
      amenities: ["wifi", "parking", "breakfast"],
    },
    {
      name: "Ganga View Hotel",
      location: "Rishikesh",
      image: "/ganga-view-hotel-rishikesh-river-facing-balcony.jpg",
      price: 5500,
      rating: 4.8,
      reviews: 890,
      amenities: ["wifi", "parking", "breakfast"],
    },
  ]

  const getAmenityIcon = (amenity: string) => {
    switch (amenity) {
      case "wifi":
        return <Wifi className="h-3 w-3" />
      case "parking":
        return <ParkingCircle className="h-3 w-3" />
      case "breakfast":
        return <Coffee className="h-3 w-3" />
      default:
        return null
    }
  }

  return (
    <>
      <script type="application/ld+json" dangerouslySetInnerHTML={jsonLd} />
      <main className="min-h-screen">
        {/* Hero Slider Section - Full viewport height */}
        <HeroSlider />

        <section className="py-6 bg-gradient-to-b from-muted/30 to-background">
          <div className="mx-auto max-w-7xl px-4 md:px-6 lg:px-8">
            <header className="text-center space-y-2 mb-6">
              <Badge variant="outline" className="inline-flex px-3 py-0.5 text-xs">
                Explore Uttarakhand
              </Badge>
              <h2 className="text-2xl md:text-3xl font-bold text-balance">
                Popular <span className="frontend-text-gradient">Destinations</span>
              </h2>
              <p className="text-muted-foreground text-sm max-w-2xl mx-auto">
                Discover the divine beauty of Dev Bhoomi - from sacred temples to thrilling adventures
              </p>
            </header>

            {destinationCategories.map((category, categoryIndex) => (
              <div key={categoryIndex} className="mb-6 last:mb-0">
                <div className="flex items-center justify-between mb-3">
                  <div className="flex items-center gap-3">
                    <div className="relative h-10 w-10 rounded-lg overflow-hidden shadow-md">
                      <Image
                        src={category.image || "/placeholder.svg"}
                        alt={category.name}
                        fill
                        className="object-cover"
                        loading="lazy"
                      />
                    </div>
                    <h3 className="text-lg font-bold">{category.name}</h3>
                  </div>
                  <Button asChild size="sm" className="frontend-btn h-8 text-xs">
                    <Link href={`/${category.name.toLowerCase()}-packages`}>
                      Book Now
                      <ArrowRight className="ml-1.5 h-3.5 w-3.5" />
                    </Link>
                  </Button>
                </div>
                <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-3">
                  {category.destinations.map((dest, destIndex) => (
                    <Card
                      key={destIndex}
                      className="group overflow-hidden border-0 shadow-lg hover:shadow-xl transition-all duration-300 cursor-pointer rounded-xl p-0"
                    >
                      <div className="relative aspect-[4/3] overflow-hidden rounded-xl">
                        <Image
                          src={dest.image || "/placeholder.svg"}
                          alt={dest.name}
                          fill
                          className="object-cover group-hover:scale-110 transition-transform duration-500"
                          loading="lazy"
                          sizes="(max-width: 640px) 50vw, (max-width: 768px) 33vw, 20vw"
                        />
                        <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/20 to-transparent" />
                        <div className="absolute bottom-0 left-0 right-0 p-2.5">
                          <h4 className="text-white font-semibold text-sm truncate">{dest.name}</h4>
                          <div className="flex items-center gap-1 text-white/80 text-xs">
                            <MapPin className="h-3 w-3" />
                            {dest.location}
                          </div>
                        </div>
                      </div>
                    </Card>
                  ))}
                </div>
              </div>
            ))}
          </div>
        </section>

        <section className="py-6 bg-gradient-to-b from-background to-muted/30">
          <div className="mx-auto max-w-7xl px-4 md:px-6 lg:px-8">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-xl md:text-2xl font-bold">
                Book Your <span className="frontend-text-gradient">Perfect Ride</span>
              </h2>
              <Button
                asChild
                variant="outline"
                size="sm"
                className="border-primary text-primary hover:bg-primary/10 bg-transparent h-8 text-xs"
              >
                <Link href="/vehicle-booking">
                  View All Vehicles
                  <ArrowRight className="ml-1.5 h-3.5 w-3.5" />
                </Link>
              </Button>
            </div>

            <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-3">
              {vehicles.map((vehicle, index) => (
                <Card
                  key={index}
                  className="group overflow-hidden border-0 shadow-lg hover:shadow-xl transition-all duration-300 rounded-xl p-0"
                >
                  <div className="relative aspect-[4/3] overflow-hidden">
                    <Image
                      src={vehicle.image || "/placeholder.svg"}
                      alt={vehicle.name}
                      fill
                      className="object-cover group-hover:scale-105 transition-transform duration-500"
                      loading="lazy"
                      sizes="(max-width: 640px) 50vw, (max-width: 768px) 33vw, 20vw"
                    />
                    <div className="absolute top-2 right-2">
                      <Badge className="bg-white/90 text-foreground text-[10px] font-medium shadow-sm px-1.5 py-0.5">
                        <Star className="h-2.5 w-2.5 fill-yellow-500 text-yellow-500 mr-0.5" />
                        {vehicle.rating}
                      </Badge>
                    </div>
                  </div>
                  <CardContent className="p-2 space-y-1">
                    <div>
                      <h4 className="font-semibold text-xs truncate">{vehicle.name}</h4>
                      <p className="text-[10px] text-muted-foreground">{vehicle.type}</p>
                    </div>
                    <div className="flex items-center justify-between text-[10px] text-muted-foreground">
                      <span className="flex items-center gap-0.5">
                        <Users className="h-2.5 w-2.5" /> {vehicle.seats}
                      </span>
                      <span className="flex items-center gap-0.5">
                        <Fuel className="h-2.5 w-2.5" /> {vehicle.fuel}
                      </span>
                    </div>
                    <div className="pt-1 border-t border-border/50 flex items-center justify-between">
                      <div>
                        <span className="text-primary font-bold text-sm">₹{vehicle.price.toLocaleString()}</span>
                        <span className="text-[10px] text-muted-foreground">/day</span>
                      </div>
                      <Button size="sm" variant="outline" className="h-6 text-[10px] px-2 bg-transparent">
                        Book
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </section>

        <section className="py-6 bg-gradient-to-b from-muted/30 to-background">
          <div className="mx-auto max-w-7xl px-4 md:px-6 lg:px-8">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-xl md:text-2xl font-bold">
                Featured <span className="frontend-text-gradient">Hotels & Resorts</span>
              </h2>
              <Button
                asChild
                variant="outline"
                size="sm"
                className="border-primary text-primary hover:bg-primary/10 bg-transparent h-8 text-xs"
              >
                <Link href="/hotel-booking">
                  View All Hotels
                  <ArrowRight className="ml-1.5 h-3.5 w-3.5" />
                </Link>
              </Button>
            </div>

            <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-3">
              {hotels.map((hotel, index) => (
                <Card
                  key={index}
                  className="group overflow-hidden border-0 shadow-lg hover:shadow-xl transition-all duration-300 rounded-xl p-0"
                >
                  <div className="relative aspect-[4/3] overflow-hidden">
                    <Image
                      src={hotel.image || "/placeholder.svg"}
                      alt={hotel.name}
                      fill
                      className="object-cover group-hover:scale-105 transition-transform duration-500"
                      loading="lazy"
                      sizes="(max-width: 640px) 50vw, (max-width: 768px) 33vw, 20vw"
                    />
                    <div className="absolute top-2 right-2">
                      <Badge className="bg-white/90 text-foreground text-[10px] font-medium shadow-sm px-1.5 py-0.5">
                        <Star className="h-2.5 w-2.5 fill-yellow-500 text-yellow-500 mr-0.5" />
                        {hotel.rating}
                      </Badge>
                    </div>
                  </div>
                  <CardContent className="p-2 space-y-1">
                    <div>
                      <h4 className="font-semibold text-xs truncate">{hotel.name}</h4>
                      <div className="flex items-center gap-0.5 text-[10px] text-muted-foreground">
                        <MapPin className="h-2.5 w-2.5" />
                        {hotel.location}
                      </div>
                    </div>
                    <div className="flex items-center gap-1.5">
                      {hotel.amenities.map((amenity, i) => (
                        <span
                          key={i}
                          className="flex items-center justify-center h-4 w-4 rounded bg-muted text-muted-foreground"
                        >
                          {getAmenityIcon(amenity)}
                        </span>
                      ))}
                      <span className="text-[10px] text-muted-foreground">({hotel.reviews})</span>
                    </div>
                    <div className="pt-1 border-t border-border/50 flex items-center justify-between">
                      <div>
                        <span className="text-primary font-bold text-sm">₹{hotel.price.toLocaleString()}</span>
                        <span className="text-[10px] text-muted-foreground">/night</span>
                      </div>
                      <Button size="sm" variant="outline" className="h-6 text-[10px] px-2 bg-transparent">
                        Book
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </section>

        <section className="py-8 bg-gradient-to-b from-background to-muted/30">
          <div className="mx-auto max-w-7xl px-4 md:px-6 lg:px-8">
            <header className="text-center space-y-2 mb-6">
              <h2 className="text-xl md:text-2xl font-bold text-balance">
                Why Choose <span className="frontend-text-gradient">Teertham</span>
              </h2>
            </header>
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-3">
              {advantages.map((advantage, index) => {
                const Icon = advantage.icon
                return (
                  <Card key={index} className="border-0 shadow-lg hover:shadow-xl transition-shadow rounded-xl">
                    <CardContent className="p-3 text-center space-y-2">
                      <div className="flex justify-center">
                        <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-primary/10">
                          <Icon className="h-5 w-5 text-primary" />
                        </div>
                      </div>
                      <h3 className="font-semibold text-xs">{advantage.title}</h3>
                      <p className="text-[10px] text-muted-foreground line-clamp-2">{advantage.description}</p>
                    </CardContent>
                  </Card>
                )
              })}
            </div>
          </div>
        </section>

        <section className="py-4 bg-gradient-to-b from-muted/30 to-background">
          <div className="mx-auto max-w-7xl px-4 md:px-6 lg:px-8">
            <Card className="overflow-hidden border-0 shadow-xl rounded-xl">
              <div className="relative">
                <div className="absolute inset-0 frontend-gradient-primary opacity-95" />
                <CardContent className="relative px-4 md:px-6 py-2 text-center space-y-1.5">
                  <h2 className="text-xl md:text-2xl font-bold text-white text-balance">
                    Ready to Start Your Journey?
                  </h2>
                  <p className="text-white/90 max-w-xl mx-auto text-sm">
                    Join thousands of satisfied travelers. Book your perfect trip today.
                  </p>
                  <div className="flex flex-col sm:flex-row gap-2 justify-center">
                    <Button asChild size="lg" className="bg-white text-primary hover:bg-white/90 h-9 text-sm shadow-lg">
                      <Link href="/create-account">
                        Get Started Free
                        <ArrowRight className="ml-2 h-4 w-4" />
                      </Link>
                    </Button>
                    <Button
                      asChild
                      size="lg"
                      variant="outline"
                      className="border-white/50 text-white hover:bg-white/10 bg-transparent h-9 text-sm"
                    >
                      <Link href="/spiritual-packages">Explore Packages</Link>
                    </Button>
                  </div>
                </CardContent>
              </div>
            </Card>
          </div>
        </section>
      </main>
    </>
  )
}
